///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file XFormManager.h
 * \brief Contains the definition of the Core::XFormManager class.
 */

#ifndef __OVITO_XFORM_MANAGER_H
#define __OVITO_XFORM_MANAGER_H

#include <core/Core.h>
#include "SelectionMode.h"
#include "MoveMode.h"
#include "RotationMode.h"
#include "ScalingMode.h"

namespace Core {

class SceneNode;	// defined in SceneNode.h
class Viewport;		// defined in Viewport.h

/// \def XFORM_MANAGER
/// \brief The predefined instance of the Core::XFormManager class.
///
/// Always use this macro to access the Core::XFormManager class instance.
#define XFORM_MANAGER		(*XFormManager::getSingletonInstance())

/**
 * \brief This class manages the settings used by the transformation input modes.
 *
 * This is a singleton class with only one predefined instance of this class.
 * You can access the instance of this class using the XFORM_MANAGER macro.
 *
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT XFormManager : public QObject
{
	Q_OBJECT

public:

	/// \brief Specifies the coordinate system to be used for an object transformation.
	enum XFormSystem {
		CONSTRUCTION_GRID_SYS,	///< The coordinate system defined by the viewport's construction grid.
		WORLD_SYS,				///< The absolute world coordinate system.
		LOCAL_SYS,				///< The local SceneNode coordinate system.
		SCREEN_SYS				///< The viewport screen coordinate system (Viewport::viewMatrix())
	};

	/// \brief Specifies the center point for an object transformation.
	enum XFormCenterMode {
		SELECTION_CENTER,		///< Use the center of the bounding box of the selection set as transformation center.
		LOCAL_ORIGIN,			///< Use the origin of the coordinate system of the SceneNode to be transformed as transformation center.
		SYSTEM_ORIGIN			///< Use the origin of the current XFormSystem as transformation center.
	};

public:

	/// \brief Returns the one and only instance of this class.
	/// \return The predefined instance of the XFormManager singleton class.
	/// \note You should use the XFORM_MANAGER macro to access the XFormManager instance instead
	///       of this method.
	inline static XFormManager* getSingletonInstance() {
		OVITO_ASSERT_MSG(_singletonInstance != NULL, "XFormManager::getSingletonInstance", "XFormManager class is not initialized yet.");
		return _singletonInstance;
	}

	/// \brief Gets the coordinate system to use for transformation of scene nodes.
	/// \return The current coordinate system used for transformations.
	/// \sa setXFormSystem()
	XFormSystem xformSystem() const { return _xformSystem; }

	/// \brief Sets the coordinate system to use for transformation of scene nodes.
	/// \param sys The new coordinate system to be used for transformations.
	///
	/// This generates a xformSystemChanged() notification event.
	/// \sa xformSystem()
	void setXFormSystem(XFormSystem sys);

	/// \brief Gets the center mode to use for transformation of scene nodes.
	/// \return The current transformation center mode.
	///
	/// The center mode controls which point to use as origin for scaling and rotation of scene objects.
	/// \sa setCenterMode()
	XFormCenterMode centerMode() const { return _centerMode; }

	/// \brief Sets the center mode to use for transformation of scene nodes.
	/// \param mode The new transformation center mode.
	///
	/// The center mode controls which point to use as origin for scaling and rotation of nodes.
	///
	/// This generates a xformCenterChanged() notification event.
	/// \sa centerMode()
	void setCenterMode(XFormCenterMode mode);

	/// \brief Returns the coordinate system to use for the xform manipulation modes.
	/// \param[in] contextNode The scene node whose transformation system should be returned.
	/// \param[out] sysMat The matrix to be filled with the return value.
	///
	/// Translations, rotations and scaling of the given node should be done in the returned coordinate system.
	void getTransformationSystem(SceneNode* contextNode, AffineTransformation& sysMat);

	/// \brief Returns the origin of the transformation system to use for xform manipulation modes.
	/// \param contextNode The scene node for which the transformation center should be returned.
	/// \return The origin of the transformation system in world coordinates.
	Point3 getTransformationCenter(SceneNode* contextNode);

	/// \brief Renders the tripods for all selected nodes in the scene.
	/// \param vp The viewport into which the tripods should be rendered.
	///
	/// This method is called by the current SceneRenderer.
	void renderTripods(Viewport* vp);

	/// \brief Renders a single tripod into the viewport.
	/// \param vp The viewport into which the tripod should be rendered.
	/// \param tm The coordinate system of the tripod.
	/// \param onlyRotationComponent Controls whether only the rotational component of the transformation matrix is used to
	///        transform the tripod.
	void renderTripod(Viewport* vp, const AffineTransformation& tm, bool onlyRotationComponent = true);

	/// \brief Returns a pointer the standard object selection mode.
	/// \return The viewport input handler that lets the user select objects in the scene.
	SelectionMode* objectSelectionMode() const { return _objectSelectionMode.get(); }

	/// \brief Returns a pointer the standard object move input mode.
	/// \return The viewport input handler that lets the user move objects in the scene.
	MoveMode* objectMoveMode() const { return _objectMoveMode.get(); }

	/// \brief Returns a pointer the standard object rotation input mode.
	/// \return The viewport input handler that lets the user rotate objects in the scene.
	RotationMode* objectRotationMode() const { return _objectRotationMode.get(); }

	/// \brief Returns a pointer the standard object scaling input mode.
	/// \return The viewport input handler that lets the user scale objects in the scene.
	ScalingMode* objectScalingMode() const { return _objectScalingMode.get(); }

public Q_SLOTS:

	/// \brief Resets the transformation settings to their default values.
	void reset();

Q_SIGNALS:

	/// \brief This signal is emitted when the xform system has been changed.
	/// \sa XFormSystem
	/// \sa setXFormSystem()
	void xformSystemChanged();

	/// \brief This signal is emitted when the xform center has been changed.
	/// \sa XFormCenterMode
	/// \sa setCenterMode()
	void xformCenterChanged();

private:

	/// Specifies the scaling factor for the tripod.
	FloatType tripodSize;

	/// Specifies the relative size of the tripod arrows.
	FloatType tripodArrowSize;

	/// Stores the current coordinate system to use for transformations.
	XFormSystem _xformSystem;

	/// Stores the current transformation center mode that determines the way how the origin
	/// of the transformation system is chosen.
	XFormCenterMode _centerMode;

private:

	/// Private constructor.
	/// This is a singleton class; no public instances are allowed.
	XFormManager();

	/// The global instance of the standard object selection mode.
	intrusive_ptr<SelectionMode> _objectSelectionMode;
	/// The global instance of the standard object move mode.
	intrusive_ptr<MoveMode> _objectMoveMode;
	/// The global instance of the standard object rotation mode.
	intrusive_ptr<RotationMode> _objectRotationMode;
	/// The global instance of the standard object scaling mode.
	intrusive_ptr<ScalingMode> _objectScalingMode;

	/// Initializes the XFormManager.
	/// This is called at program startup.
	static void initialize() {
		OVITO_ASSERT(_singletonInstance == NULL);
		_singletonInstance = new XFormManager();
	}

	/// XFormManager shutdown.
	static void shutdown() {
		delete _singletonInstance;
		_singletonInstance = NULL;
	}

	/// The singleton instance of this class.
	static XFormManager* _singletonInstance;

	friend class ApplicationManager;
};

};

#endif // __OVITO_XFORM_MANAGER_H
