# -*- coding: utf-8 -*-
import pygame
import colorsys
import os, sys
import random
import copy
import pygame.mixer
import classes.extras as ex
    
sounds = pygame.mixer
sounds.init()
sound_8 = '104874__robinhood76__02020-cartoon-slide_1.ogg'
sound_6 = '104874__robinhood76__02020-cartoon-slide_2.ogg'
s1 = sounds.Sound(os.path.join('sounds', sound_8))
s2 = sounds.Sound(os.path.join('sounds', sound_6))

class Unit(pygame.sprite.Sprite):
    'basic class for all on-board objects'
    def __init__(self,board,grid_x=0,grid_y=0,grid_w=1,grid_h=1,value="",color=(0,0,0),**kwargs):
        # Call the parent's constructor
        pygame.sprite.Sprite.__init__(self)

        #grid location and size
        self.grid_x = grid_x
        self.grid_y = grid_y
        self.grid_w = grid_w
        self.grid_h = grid_h
        self.board = board
        self.initcolor = color 
        self.color = color
        self.locked = False
        self.lockable = False
        self.value = value
        self.speaker_val = value
        self.speaker_val_update = True
        self.outline = False
        self.perm_outline = False
        self.perm_outline_color = [255,0,0]
        self.perm_outline_width = 2
        self.hasimg = False
        self.draggable = True
        self.keyable = True
        self.show_value = True
        self.readable = True
        self.highlight = True
        self.audible = False #use true to enable sounds on unit move
        self.outline_highlight = False
        self.font_color = (0,0,0,0)
        self.align = 0 #align: 0 - centered, 1 - left, 2 - right
        self.update_me = True
        # Set height, width, the -1 is to give it some space around for the margin
        self.image = pygame.Surface([grid_w*board.scale-1, grid_h*board.scale-1])
        self.image.fill(self.color)
        
        #http://www.pygame.org/docs/ref/surface.html - surface.fill() comment
        #self.image = pygame.Surface([grid_w*board.scale-1, grid_h*board.scale-1],flags=pygame.SRCALPHA)
        #self.image.fill(self.color,special_flags=pygame.BLEND_RGBA_MIN)
        self.painting = self.image
 
        # Make our top-left corner the passed-in location. The +1 is the margin
        self.rect = self.image.get_rect()
        self.rect.topleft = [grid_x*board.scale+1,grid_y*board.scale+1]

        #scale font size:
        self.font = board.font_sizes[0]
        
    def pos_update(self):
        if self.grid_w > 0 and self.grid_h > 0:
            self.image = pygame.Surface([self.grid_w*self.board.scale-1, self.grid_h*self.board.scale-1])
            self.painting = self.image            
            self.rect = self.image.get_rect()
            self.rect.topleft = [self.grid_x*self.board.scale+1,self.grid_y*self.board.scale+1]
        else:
            self.image = pygame.Surface([1,1])
            #self.painting = self.image            
            self.rect = self.image.get_rect()
            self.rect.topleft = [self.grid_x*self.board.scale+1,self.grid_y*self.board.scale+1]
            
    def scale_img(self, new_w, new_h):
        'scales image depending on pygame version and bit depth using either smoothscale or scale' 
        if self.img.get_bitsize() in [32,24] and pygame.version.vernum >= (1, 8):
            self.img = self.img_org = pygame.transform.smoothscale(self.img, (new_w, new_h))
        else:
            self.img = self.img_org = pygame.transform.scale(self.img, (new_w, new_h))
        
    @property
    def grid_pos(self):
        return [self.grid_x, self.grid_y]

    def immobilize(self):
        self.keyable = False
        self.draggable = False
        self.highlight = False
        
    # Update color, image or text
    def update(self, board, **kwargs):
        #self.update_me = True
        if self.update_me:
            self.update_me = False
            self.image.fill(self.color)
            self.image.blit(self.painting,(0,0))
            if self.hasimg == False:
                if len(self.value) > 0:
                    if self.show_value:
                        if sys.version_info < (3, 0):
                            if isinstance(self.value, basestring):
                                #if a passed argument is a string turn it into a 1 item list
                                value = [self.value]
                            else:
                                value = self.value
                        else:
                            if isinstance(self.value, str):
                                #if a passed argument is a string turn it into a 1 item list
                                value = [self.value]
                            else:
                                value = self.value

                        lv = len(value)
                        for i in range(lv):
                            if sys.version_info < (3, 0):
                                try:
                                    val = unicode(value[i], "utf-8")
                                except UnicodeDecodeError:
                                    val = value[i]
                                except TypeError:
                                    val = value[i]
                            else:
                                val = value[i]
                            text = self.font.render("%s" % (val), 1, self.font_color)
    
                            if self.align == 0:
                                font_x = ((board.scale*self.grid_w-self.font.size(val)[0])//2)
                            elif self.align == 1:
                                font_x = 5
                            elif self.align == 2:
                                font_x = board.scale*self.grid_w - self.font.size(val)[0]-5
    
                            if lv == 1:
                                font_y = ((board.scale*self.grid_h-self.font.size(val)[1])//2)
                            elif lv == self.grid_h: #number of items is equal to grid height of an object - distribute lines equally in each grid square
                                font_y = ((board.scale-self.font.size(val)[1])//2)+board.scale*i
                            else:
                                #lv - total
                                line_h = self.font.size(value[0])[1]
                                line_margin = 0 #(board.scale*self.grid_h - line_h*lv)//lv #self.font.size(value[0])[1]//4
                                step = line_h + line_margin
                                center = (board.scale*self.grid_h)//2
                                start_at = center - (step*lv - line_margin)//2
                                font_y = start_at + step*i                        
                            self.image.blit(text, (font_x,font_y))
                            
            if self.speaker_val_update:
                self.speaker_val = self.value          
    
            if self.perm_outline:
                self.draw_outline()
                    
    @property
    def reversed_color(self):
        return [int(each / 1.5) for each in reversed(self.initcolor)]

    @property
    def brighter(self):
        if self.highlight:
            color = [each / 255.0 for each in self.initcolor]
            hsv = colorsys.rgb_to_hsv(*color)
            rgb = colorsys.hsv_to_rgb(hsv[0],1,1)
            return [int(each*255) for each in rgb]
        else:
            return self.initcolor

    def turn(self,d):
        pass
     
    def rot_center(self, image, angle):
        """rotate an image while keeping its center and size"""
        orig_rect = image.get_rect()
        rot_image = pygame.transform.rotate(image, angle)
        rot_rect = copy.deepcopy(orig_rect)
        rot_rect.center = rot_image.get_rect().center
        rot_image = copy.copy(rot_image.subsurface(rot_rect))
        return rot_image
        
    def draw_outline(self):
        "draws an 'outline' around the unit"
        color = self.perm_outline_color#[255,0,0]
        width = self.perm_outline_width
        if width > 1:
            x = width // 2-1
            y = width // 2-1
            if width % 2 == 0:
                w2 = width // 2 + 2
            else:
                w2 = width // 2 + 1
        elif width == 1:
            x = 0
            y = 0
            w2 = 2
        pygame.draw.lines(self.image, color, True, [[x-width,y],[self.board.scale*self.grid_w-w2+width,y],[self.board.scale*self.grid_w-w2,y-width],[self.board.scale*self.grid_w-w2,self.board.scale*self.grid_h-w2+width],[self.board.scale*self.grid_w-w2+width,self.board.scale*self.grid_h-w2],[x-width,self.board.scale*self.grid_h-w2], [x,self.board.scale*self.grid_h-w2+width],[x,y-width]],width)  

        
    def set_outline(self, color = [255,0,0], width = 2):
        'enables the draw_outline and sets line color and width'
        self.perm_outline = True
        if color == 0 and hasattr(self,"door_outline") == False: #if color is 0 calculate colour from base colour
            #convert to hsv
            c = self.color
            h, s, v = ex.rgb_to_hsv(c[0],c[1],c[2])
            outline_color =  ex.hsv_to_rgb(h,s+50,v-50)
            self.perm_outline_color = outline_color
        elif color == 1:
            c = self.color
            h, s, v = ex.rgb_to_hsv(c[0],c[1],c[2])
            outline_color =  ex.hsv_to_rgb(h,s+20,v-20)
            self.perm_outline_color = outline_color
        elif hasattr(self,"door_outline") == False:
            self.perm_outline_color = color
        else:
            pass
        #self.perm_outline_color = color
        #self.perm_outline_color = [255,0,0]
        self.perm_outline_width = width
        self.init_pow = width
        
class Obstacle(Unit):
    def __init__(self,board,grid_x=0,grid_y=0,grid_w=1,grid_h=1,value="", initcolor = (23,157,255),**kwargs):
        self.initcolor = initcolor
        Unit.__init__(self,board,grid_x,grid_y,grid_w,grid_h,"0",self.initcolor,**kwargs)
        self.unit_id = len(board.units)
        self.value = value

class Label(Obstacle):
    def __init__(self,board,grid_x=0,grid_y=0,grid_w=1,grid_h=1,value="",initcolor = (255,157,23),font_size=0,**kwargs):
        Obstacle.__init__(self,board,grid_x,grid_y,grid_w,grid_h,value,initcolor,**kwargs)
        self.font = board.font_sizes[font_size]
            
    def update(self, board, **kwargs):
        Unit.update(self,board)

class Ship(Unit):
    def __init__(self,board,grid_x=0,grid_y=0,grid_w=1,grid_h=1,value="",initcolor = (255,157,23),**kwargs):
        self.initcolor = initcolor
        Unit.__init__(self,board,grid_x,grid_y,grid_w,grid_h,value,self.initcolor,**kwargs)
        self.unit_id = len(board.ships)
        
    def move(self,board,x,y):
        board.move(self.unit_id,x,y)

    def update(self, board, point,**kwargs):
        if self.update_me:
            Unit.update(self,board)
            if self.lockable and self.locked:
                self.draw_circle(board, point)

    def enable_circle(self):
        self.locked = True

    def disable_circle(self):
        self.locked = False
        
    def draw_circle(self,board,point):
        max_radius = board.scale // 2
        step = max_radius // 4
        color = self.reversed_color
        for i in range(1,4):
            try:
                pygame.draw.ellipse(self.image, color, ((point[0]+step,point[1]+step),(board.scale-step*2,board.scale-step*2)), 1)
                step += step
            except ValueError:
                pass
        pygame.draw.line(self.image, color,[point[0]+board.scale//2,point[1]],[point[0]+board.scale//2,point[1]+board.scale],1)
        pygame.draw.line(self.image, color,[point[0],point[1]+board.scale//2],[point[0]+board.scale,point[1]+board.scale//2],1)

        
class Letter(Ship):
    def __init__(self,board,grid_x=0,grid_y=0,grid_w=1,grid_h=1,value="",initcolor = (255,157,23),font_size=0,**kwargs):
        Ship.__init__(self,board,grid_x,grid_y,grid_w,grid_h,value,initcolor,**kwargs)
        self.font = board.font_sizes[font_size]

    def update(self, board, **kwargs):
        Unit.update(self,board)   
             
class ImgSurf(pygame.sprite.Sprite):
    def __init__(self,board,grid_w=1,grid_h=1,color = (255,157,23), img_src=''):
        pygame.sprite.Sprite.__init__(self)        
        #Ship.__init__(self,board,grid_x,grid_y,grid_w,grid_h,value,initcolor,**kwargs)
        self.img_src = img_src
        #grid location and size
        self.grid_w = grid_w
        self.grid_h = grid_h
        self.board = board
        self.color = color
        self.image = pygame.Surface([grid_w*board.scale-1, grid_h*board.scale-1])
        self.image.fill(self.color)
        self.rect = self.image.get_rect()
        
        if len(self.img_src) > 0:
            self.hasimg = True
            self.img = self.image
            self.img_pos = (0,0)
            self.outline = True
            try:
                self.img_org = pygame.image.load(os.path.join('images', self.img_src)).convert()
                self.img = self.img_org
                self.img_rect = self.img.get_rect()
                #resize the image
                self.scale_img(self.rect.w, self.rect.h)

                self.img_rect = self.img.get_rect()                
                pos_x = ((board.scale*self.grid_w-self.img_rect.w)//2)
                pos_y = ((board.scale*self.grid_h-self.img_rect.h)//2)
                self.img_pos = (pos_x,pos_y)
            except IOError:
                pass

    def scale_img(self, new_w, new_h):
        'scales image depending on pygame version and bit depth using either smoothscale or scale' 
        if self.img.get_bitsize() in [32,24] and pygame.version.vernum >= (1, 8):
            self.img = self.img_org = pygame.transform.smoothscale(self.img, (new_w, new_h))
        else:
            self.img = self.img_org = pygame.transform.scale(self.img, (new_w, new_h))
        
class ImgShip(Ship):
    def __init__(self,board,grid_x=0,grid_y=0,grid_w=1,grid_h=1,value="",initcolor = (255,157,23), img_src='',**kwargs):
        Ship.__init__(self,board,grid_x,grid_y,grid_w,grid_h,value,initcolor,**kwargs)
        self.img_src = img_src
        if len(self.img_src) > 0:
            self.hasimg = True
            self.img = self.image
            self.img_pos = (0,0)
            self.outline = True
            try:
                self.img_org = pygame.image.load(os.path.join('images', self.img_src)).convert()
                self.img = self.img_org
                self.img_rect = self.img.get_rect()
                #resize the image
                self.scale_img(self.rect.w, self.rect.h)

                self.img_rect = self.img.get_rect()                
                pos_x = ((board.scale*self.grid_w-self.img_rect.w)//2)
                pos_y = ((board.scale*self.grid_h-self.img_rect.h)//2)
                self.img_pos = (pos_x,pos_y)
            except IOError:
                pass
        #bg_color = tuple(self.image.get_at((5,5)))
        #self.image.set_colorkey(bg_color[:3])
        #self.image.set_colorkey(self.initcolor)
        #Surface.get_at((x, y)): return Color

    def update(self, board, **kwargs):
        if self.update_me:
            Unit.update(self,board)
            if len(self.img_src) > 0:
                self.image.blit(self.img, self.img_pos)
            if self.unit_id == board.active_ship and self.outline == True:
                lines = [[0,0],[self.grid_w*board.scale-2,0],[self.grid_w*board.scale-2,self.grid_h*board.scale-2],[0,self.grid_h*board.scale-2]]
                pygame.draw.lines(self.image, (255, 200, 200), True, lines)
            if hasattr(self, "door_outline") and self.door_outline == True:
                self.set_outline(self.perm_outline_color,2)
            if self.perm_outline:
                self.draw_outline()
    
    @property
    def brighter(self):
        return self.color

class ImgAlphaShip(ImgShip):
    def __init__(self,board,grid_x=0,grid_y=0,grid_w=1,grid_h=1,value="",initcolor = (255,157,23), img_src='',**kwargs):
        Ship.__init__(self,board,grid_x,grid_y,grid_w,grid_h,value,initcolor,**kwargs)
        self.img_src = img_src
        if len(self.img_src) > 0:
            self.hasimg = True
            self.img = self.image
            self.img_pos = (0,0)
            self.outline = True
            try:
                self.img_org = pygame.image.load(os.path.join('images', self.img_src)).convert_alpha()
                self.img = self.img_org
                self.img_rect = self.img.get_rect()
                #resize the image
                self.scale_img(self.rect.w, self.rect.h)

                self.img_rect = self.img.get_rect()                
                pos_x = ((board.scale*self.grid_w-self.img_rect.w)//2)
                pos_y = ((board.scale*self.grid_h-self.img_rect.h)//2)
                self.img_pos = (pos_x,pos_y)
            except IOError:
                pass

        #self.image.set_colorkey(self.initcolor)

class ImgCenteredShip(Ship):
    def __init__(self,board,grid_x=0,grid_y=0,grid_w=1,grid_h=1,value="",initcolor = (255,157,23), img_src='',**kwargs):
        Ship.__init__(self,board,grid_x,grid_y,grid_w,grid_h,value,initcolor,**kwargs)
        self.img_src = img_src
        if len(self.img_src) > 0:
            self.hasimg = True
            self.img = self.image
            self.img_pos = (0,0)
            self.outline = True
            try:
                self.img_org = pygame.image.load(os.path.join('images', self.img_src)).convert()
                self.img = self.img_org
                self.img_rect = self.img.get_rect()
                old_h = self.img_rect.h
                old_w = self.img_rect.w
                
                new_h = self.rect.h
                new_w = int((new_h*old_w)/old_h)
                #resize the image
                self.scale_img(new_w, new_h)

                self.img_rect = self.img.get_rect()                
                pos_x = ((board.scale*self.grid_w-self.img_rect.w)//2)
                pos_y = ((board.scale*self.grid_h-self.img_rect.h)//2)
                self.img_pos = (pos_x,pos_y)
            except IOError:
                pass

        #self.image.set_colorkey(self.initcolor)

    def update(self, board, **kwargs):
        if self.update_me:
            Unit.update(self,board)
            if len(self.img_src) > 0:
                self.image.blit(self.img, self.img_pos)
            if self.unit_id == board.active_ship and self.outline == True:
                lines = [[0,0],[self.grid_w*board.scale-2,0],[self.grid_w*board.scale-2,self.grid_h*board.scale-2],[0,self.grid_h*board.scale-2]]
                pygame.draw.lines(self.image, (255, 200, 200), True, lines)
            if hasattr(self, "door_outline") and self.door_outline == True:
                #self.set_outline([255,0,0],2)
                self.set_outline(self.perm_outline_color,self.perm_outline_width)
            if self.perm_outline:
                self.draw_outline()

class MultiImgSprite(ImgShip):
    def __init__(self,board,grid_x=0,grid_y=0,grid_w=1,grid_h=1,value="",initcolor = (255,157,23), img_src='',frame_flow=[0], frame_count=1,row_data=[1,1], **kwargs):
        ImgShip.__init__(self,board,grid_x,grid_y,grid_w,grid_h,value,initcolor,**kwargs)
        self.img_src = img_src
        if len(self.img_src) > 0:
            self.hasimg = True
            self.img = self.image
            self.img_pos = (0,0)
            self.outline = False
            self.draggable=False
            self.correction = False
            self.frame_w = grid_w*board.scale
            self.frame_h = grid_h*board.scale
            self.frame_flow = frame_flow
            self.frame_count = frame_count
            self.row_data = row_data #[number of images per row,number of rows]
            self.correction_factor = 3.0
            
            self.frame = 0
            try:
                self.img_org = pygame.image.load(os.path.join('images', self.img_src)).convert()
                self.img = self.img_org
                self.img_rect = self.img.get_rect()
                
                #image size is most likely different than the sprite so resize is needed
                new_h = self.rect.h * self.row_data[1]
                new_w = new_h * self.img_rect.w // self.img_rect.h
                self.scale_img(new_w, new_h)                
                self.img_rect = self.img.get_rect()                
                pos_x = 0
                pos_y = 0
                self.img_pos = (pos_x,pos_y)
            except IOError:
                pass

    def next_frame(self):
        if self.frame < self.frame_count-1:
            self.frame += 1
        else:
            self.frame = 0
        xg = self.frame_flow[self.frame] % self.row_data[0]
        yg = self.frame_flow[self.frame] // self.row_data[0]
        x = -(xg*(self.frame_w-1))
        y = -(yg*(self.frame_h-1))
        self.img_pos = (x,y)
    
    def set_frame(self, frame):
        self.frame = frame

        xg = self.frame_flow[self.frame] % self.row_data[0]
        yg = self.frame_flow[self.frame] // self.row_data[0]
        if self.correction:
            #shift the image by 1px to the right every x frames - to avoid scaling problem with very long images
            shift_x = int(float(xg) / self.correction_factor)
            shift_y = int(float(yg) / self.correction_factor) #int(float(self.frame_flow[self.frame]) / self.correction_factor)
        else:
            shift_x = 0
            shift_y = 0
        x = -(xg*(self.frame_w-1))+shift_x
        y = -(yg*(self.frame_h-1))+shift_y
        self.img_pos = (x,y)
        
        #self.img_pos = (-(self.frame_flow[self.frame]*(self.frame_w-1))+shift,0)
    
    def build_frame_flow(self,frame_count,frame_flow = []):
        self.frame_count = frame_count
        if len(frame_flow) == 0:
            self.frame_flow = [i for i in range(self.frame_count)]
        else:
            self.frame_flow = frame_flow
        
    def reset(self):
        self.img_pos = (0,0)
        self.frame = 0
        
class Door(ImgShip):
    def __init__(self,board,grid_x,grid_y,grid_w,grid_h,value,initcolor,font_size,**kwargs):
        ImgShip.__init__(self,board,grid_x,grid_y,grid_w,grid_h,value,initcolor,**kwargs)
        self.font = board.font_sizes[font_size]
        self.image.set_colorkey(self.initcolor)
        
    def set_pos(self,pos):
        self.grid_x = pos[0]
        self.grid_y = pos[1]
        self.rect.topleft = [pos[0]*self.board.scale+1,pos[1]*self.board.scale+1]

class SlidingDoor(MultiImgSprite):
    def set_pos(self,pos):
        self.grid_x = pos[0]
        self.grid_y = pos[1]
        self.rect.topleft = [pos[0]*self.board.scale+1,pos[1]*self.board.scale+1]

        
class PickUp(ImgShip):
    def __init__(self,board,grid_x=0,grid_y=0,grid_w=1,grid_h=1,value="",initcolor = (255,255,255), img_src='',**kwargs):
        ImgShip.__init__(self,board,grid_x,grid_y,grid_w,grid_h,value,initcolor,**kwargs)
        door_outline = False

            
class ImgShipRota(ImgShip):
    def turn(self,d):
        if d == [0,-1]:#up
            self.img = self.img_org
            self.update_me = True
        elif d == [0,1]:#down
            self.img = self.rot_center(self.img_org, 180)
            self.update_me = True
        elif d == [1,0]: #right
            self.img = self.rot_center(self.img_org, 270)
            self.update_me = True
        elif d == [-1,0]: #left
            self.img = self.rot_center(self.img_org, 90)
            self.update_me = True

class AIUnit(ImgShipRota):
    def __init__(self,board,grid_x=0,grid_y=0,grid_w=1,grid_h=1,value="",initcolor = (255,157,23),**kwargs):
        self.initcolor = initcolor
        ImgShipRota.__init__(self,board,grid_x,grid_y,grid_w,grid_h,value,self.initcolor,**kwargs)
        self.unit_id = len(board.aiunits)
        self.prev_pos = [grid_x,grid_y]
        self.now_pos  = [grid_x,grid_y]
        self.change_dir([[0,1],[1,0],[0,-1],[-1,0]])
        
    def change_dir(self, possible):
        #possible_dirs = possible #[[0,1],[1,0],[0,-1],[-1,0]]
        self.move_dir = possible[random.randrange(len(possible))]
        
class BoardBg(Unit):
    #def update(self,screen,color,screen_w,screen_h,grid_line_w):
    def __init__(self,board,grid_x=0,grid_y=0,grid_w=1,grid_h=1,value="", initcolor = (255,255,255),**kwargs):
        #self.initcolor = initcolor
        Unit.__init__(self,board,grid_x,grid_y,grid_w,grid_h,"",initcolor,**kwargs)
        self.rect.topleft = [0,0]
        #game,self.cl_grid_line,l.screen_w-l.menu_w,l.game_h,l.grid_line_w
        self.line_color = (240, 240, 240)#gb.cl_grid_line
        self.screen_w = self.board.x_count * self.board.scale # gb.l.screen_w-gb.l.menu_w
        self.screen_h = self.board.y_count * self.board.scale
        self.grid_line_w = 1

    def update(self, board, **kwargs):
        Unit.update(self,board)
        #self.painting.fill(self.color)
        
        if self.board.draw_grid:
            for row in range(self.board.x_count+1): #draw vertical lines
                pygame.draw.line(self.painting,self.line_color,[row*self.board.scale,0],[row*self.board.scale,self.screen_h],self.grid_line_w)
            for column in range(self.board.y_count+1): #draw horizontal lines
                pygame.draw.line(self.painting,self.line_color,[0,column*self.board.scale],[self.screen_w,column*self.board.scale],self.grid_line_w)

class PuzzleTable:
    def __init__(self):
        pass
    
    def clean(self):
        pass
        
class Board:
    'Initializes and creates an empty board with the sizes given, ie. a=Board(mainloop,10,10,50)'
    def __init__(self,mainloop,x_count=10,y_count=10,scale=8):
        self.mainloop = mainloop
        self.draw_grid = True
        self.check_laby = False
        self.laby_dir = -1
        self.level_start(x_count,y_count,scale)
        self.s1 = s1
        self.s2 = s2
        
    def level_start(self,x_count,y_count,scale):
        self.grid  = []     #square availability list
        self.ships = []     #list of movable objects on board
        self.units = []     #list of non moving units
        self.aiunits = []
        self.ai_enabled = False
        self.x_count=x_count #number of columns
        self.y_count=y_count #number of rows
        self.scale = scale  #number of pixels per grid unit
        self._create_board(x_count,y_count)
        self.active_ship = -1
        self.board_changed = False
        # This is a list of 'sprites.' Each block in the program is
        # added to this list. The list is managed by a class called 'RenderPlain.'
        self.unit_list = pygame.sprite.LayeredUpdates()
        self.ship_list = pygame.sprite.LayeredUpdates()
        # This is a list of every sprite. All blocks and the player block as well.
        self.all_sprites_list = pygame.sprite.LayeredUpdates()#pygame.sprite.RenderPlain()
        #self.sprites_to_draw = pygame.sprite.RenderPlain()
        
        #scaling and creating font sizes:
        self.points = int(round((self.scale * 72 /96)*1.2,0))
        
        #sizes= [0 1    2   3    4 5    6   7    8 9   10 11 11-hw 12-hw]

        #sizes = [1.0,1.25,1.5,1.75,2.0,2.25,2.5,2.75,3.0,3.5,4.0]
        sizes = [1.25,1.5,1.75,2.0,2.25,2.5,2.75,3.0,3.5,4.0,4.75,7]
        self.font_sizes = [pygame.font.Font(os.path.join('fonts', 'FreeSansBold', 'FreeSansBold.ttf'), (int(float(self.points)/float(sizes[i])))) for i in range(len(sizes))]
        #12+ handwritten        
        h_sizes = [25,17,10,1.1,1.5,2,2.3,0.7]
        handwritten_sizes = [pygame.font.Font(os.path.join('fonts', 'pysiogameFonts', 'pysiogameHand.ttf'), (int(float(self.points)*float(h_sizes[i])))) for i in range(len(h_sizes))]
        self.font_sizes.extend(handwritten_sizes)
        #20
        self.font_sizes.append(pygame.font.Font(os.path.join('fonts', 'pysiogameFonts', 'pysiogameLatinPrint.ttf'), (int(float(self.points)*float(30)))))
        #21 - extra large normal print 
        self.font_sizes.append(pygame.font.Font(os.path.join('fonts', 'FreeSansBold', 'FreeSansBold.ttf'), (int(self.points*2.0))))   
        self.font_sizes.append(pygame.font.Font(os.path.join('fonts', 'FreeSansBold', 'FreeSansBold.ttf'), (int(self.points*1.5))))   
        #23 - mini clock sizes
        self.font_sizes.append(pygame.font.Font(os.path.join('fonts', 'FreeSansBold', 'FreeSansBold.ttf'), (int(self.points/15))))   
        self.font_sizes.append(pygame.font.Font(os.path.join('fonts', 'FreeSansBold', 'FreeSansBold.ttf'), (int(self.points/25))))  
               
        self.board_bg = BoardBg(self,0,0,x_count,y_count,"",(255,255,255))
        self.unit_list.add(self.board_bg)
        self.all_sprites_list.add(self.board_bg)
        
    def update_layout(self,scale):
        pass
            
    def clean(self):
        self.unit_list.empty()
        self.ship_list.empty()
        self.all_sprites_list.empty()
        #self.sprites_to_draw.empty()
        del(self.ships)
        del(self.units)
        del(self.aiunits)
        del(self.unit_list)
        del(self.ship_list)
        del(self.all_sprites_list)
        #del(self.sprites_to_draw)
        
    def _create_board(self,sx,sy):
        'Creates an empty board for the initialisation method'
        self.grid = [[0 for x in range(0,sx)] for y in range(0,sy)]

    def _reset_board(self):
        'Sets all fields on Board to False'
        self.grid = [[0 for x in range(0,self.x_count)] for y in range(0,self.y_count)]

    def _set(self,x,y,grid_w=1,grid_h=1,value=1):
        'Take/Reserve the position on board if True, or free position if False'
        'Before using this method use the _isfree() method first check if all squares in question are available and than go back to each field and set as True'
        x2=x+grid_w
        y2=y+grid_h
        for i in range(x,x2):
            for j in range(y,y2):
                self.grid[j][i]=value

    def _isfree(self,x,y,grid_w=1,grid_h=1):
        'check if the position is free and within board'
        x2=x+grid_w
        y2=y+grid_h

        #if position + size is within board
        if (0 <= x < x2 <= self.x_count) and (0 <= y < y2 <= self.y_count):            
            for i in range(x,x2):
                for j in range(y,y2):
                    if self.grid[j][i]==True:
                        return False
            return True
        return False

    def add_unit(self, grid_x=0, grid_y=0, grid_w=1, grid_h=1, unit_class=Ship, value="A", color=(0,0,0), img_src='',font_size=0,frame_flow=[0],frame_count=1,row_data=[1,1]):
        'adds a new unit to the board'
        if self._isfree(grid_x,grid_y,grid_w,grid_h):
            unit = unit_class(self,grid_x,grid_y,grid_w,grid_h,value,initcolor=color,img_src = img_src,font_size = font_size,frame_flow = frame_flow,frame_count=frame_count,row_data=row_data)
            if isinstance(unit, Ship):
                if isinstance(unit, AIUnit):
                    self.aiunits.append(unit)
                else:
                    self.ships.append(unit) #add a ship to the ship list
                self.ship_list.add(unit) #add the ship to the sprites list
            elif isinstance(unit, Obstacle):
                self.units.append(unit)
                self.unit_list.add(unit)
            self.all_sprites_list.add(unit)
            self._set(grid_x,grid_y,grid_w,grid_h)
        else:
            print('Sorry: position taken: (x:%d, y:%d, w:%d, h:%d), board size: %d x %d, game_id: %d, screen size: %d x %d' % (grid_x,grid_y,grid_w,grid_h,self.x_count,self.y_count,self.mainloop.m.active_game_id,self.mainloop.wn_size[0],self.mainloop.wn_size[1]))
            
    def add_door(self, grid_x=0, grid_y=0, grid_w=1, grid_h=1, unit_class=Door, value="", color=(0,0,0), img_src='', font_size = 0, door_alpha = True, frame_flow=[0], frame_count=1,row_data=[1,1]):
        #add a unit that will be drawn to the board but will not hold a square in the grid
        #this is usually a red square indicating where other squares should be dragged to complete the task
        unit = unit_class(self,grid_x,grid_y,grid_w,grid_h,value,initcolor=color,img_src = img_src,font_size = font_size,door_alpha = door_alpha,frame_flow=frame_flow,frame_count=frame_count,row_data=row_data)
        self.unit_list.add(unit)
        self.units.append(unit)
        self.all_sprites_list.add(unit)
        
    def move(self, ship_id, x, y, ai=False):
        'move the ship, diagonal move possible only if two-step non diagonal move is possible'       
        if ai == True:
            s = self.aiunits[ship_id]
        else:        
            s = self.ships[self.active_ship]
        #check direction and move if fields in that direction are free
        #set out what squares need checking if move has been taken in each direction
        up    = (s.grid_x, s.grid_y -1, s.grid_w, 1)
        down  = (s.grid_x, s.grid_y + s.grid_h, s.grid_w,1)
        left  = (s.grid_x - 1, s.grid_y, 1, s.grid_h)
        right = (s.grid_x + s.grid_w, s.grid_y, 1, s.grid_h)

        #assign 'area to check' to direction
        if   x ==  0 and y == -1:
            new_rect = up
            self.laby_dir = 2
        elif x ==  0 and y ==  1:
            new_rect = down
            self.laby_dir = 3
        elif x == -1 and y ==  0:
            new_rect = left
            self.laby_dir = 1
        elif x ==  1 and y ==  0:
            new_rect = right
            self.laby_dir = 0

        #diagonal move: prepare the 2 step move alternatives to check against        
        #alt1a -> alternative path 1 firt move: a, second move: b
        elif x <= -1 and y <= -1: #up-left
            alt1a = up
            alt1b = (s.grid_x - 1, s.grid_y-1, 1, s.grid_h)
            alt2a = left
            alt2b = (s.grid_x -1 , s.grid_y -1, s.grid_w, 1)
        elif x >=  1 and y <= -1: #up-right
            alt1a = up
            alt1b = (s.grid_x + s.grid_w, s.grid_y-1, 1, s.grid_h)
            alt2a = right
            alt2b = (s.grid_x+1, s.grid_y -1, s.grid_w, 1)
        elif x <= -1 and y >=  1: #down-left
            alt1a = down
            alt1b = (s.grid_x - 1, s.grid_y+1, 1, s.grid_h)
            alt2a = left
            alt2b = (s.grid_x-1, s.grid_y + s.grid_h, s.grid_w,1)
        elif x >=  1 and y >=  1: #down-right
            alt1a = down
            alt1b = (s.grid_x + s.grid_w, s.grid_y+1, 1, s.grid_h)
            alt2a = right
            alt2b = (s.grid_x+1, s.grid_y + s.grid_h, s.grid_w,1)
        mdir = [0,0]
        if x==0 or y==0:
            #standard move: check if positions are empty and move the unit 
            if self._isfree(*new_rect):
                self._move_unit(ship_id,ai,x,y)
            else:
                if ai == False and s.audible:
                    s2.play()
        elif x != 0 and y != 0:
            self.labi_dir = -1
            #diagonal move simple path finder: check both alternatives in turn and move if possible
            #decreased number of checks to get the direction
            if self._isfree(*alt1a): #if move up or down possible change y in first alternative
                mdir[1]=y
                if self._isfree(*alt1b): #if move left or right possible change x in first alt.
                    mdir[0]=x
                else: mdir[0]=0
            elif self._isfree(*alt2a): #else if horizontal move possible change x first
                mdir[0]=x
                if self._isfree(*alt2b): #and if second move possible change y second
                    mdir[1]=y
                else: mdir[1]=0
            else:
                if ai == False and s.audible:
                    s2.play()
            if mdir != [0,0]:
                self._move_unit(ship_id,ai,mdir[0],mdir[1])
                
    def moved(self):
        pass #print("board - moved")
        
    def _move_unit(self, ship_id, ai, x, y):
        if ai == True:
            ship = self.aiunits[ship_id]
        else:        
            ship = self.ships[ship_id]
        if self.check_laby == False or (self.check_laby == True and self.laby_dir > -1 and not self.mainloop.game_board.mylaby.get_cell(ship.grid_x , ship.grid_y).laby_doors[self.laby_dir]):
            self.laby_dir = -1
            #remove ship from board grid - take off
            self._set(ship.grid_x,ship.grid_y,ship.grid_w,ship.grid_h,False)
    
            #change position of ship in ships list
            ship.grid_x += x
            ship.grid_y += y
    
            #place the ship back on board - land
            self._set(ship.grid_x,ship.grid_y,ship.grid_w,ship.grid_h,True)
    
            #update the sprite's position 
            ship.rect.topleft = [ship.grid_x*self.scale+1,ship.grid_y*self.scale+1]
            self.board_changed = True
            if ai == False:
                self.moved()
                if ship.audible:
                    s1.play()
     
    def _place_unit(self, ship_id, pos):
        ship = self.ships[ship_id]
        #remove ship from board grid - take off
        self._set(ship.grid_x,ship.grid_y,ship.grid_w,ship.grid_h,False)

        #change position of ship in ships list
        ship.grid_x = pos[0]
        ship.grid_y = pos[1]

        #place the ship back on board - land
        self._set(ship.grid_x,ship.grid_y,ship.grid_w,ship.grid_h,True)

        #update the sprite's position 
        ship.rect.topleft = [ship.grid_x*self.scale+1,ship.grid_y*self.scale+1]
        
    def get_unit_id(self,x,y):
        for each_unit in self.units:
            if each_unit.grid_x <= x <= each_unit.grid_x + each_unit.grid_w-1 \
            and each_unit.grid_y <= y <= each_unit.grid_y + each_unit.grid_h-1:
                return each_unit.unit_id
                
    def activate_ship(self,x,y):
        'this only works on binary table'
        #unhighlight and repaint deactivated unit:
        if self.active_ship != -1:
            active = self.ships[self.active_ship]
            active.color = active.initcolor
            if active.outline_highlight:
                active.perm_outline_width = active.init_pow
            active.update_me = True
            
        #activate new unit
        for each_ship in self.ships:
            if each_ship.grid_x <= x <= each_ship.grid_x + each_ship.grid_w-1 \
            and each_ship.grid_y <= y <= each_ship.grid_y + each_ship.grid_h-1:
                self.active_ship = each_ship.unit_id
                active = self.ships[self.active_ship]
                active.color = active.brighter
                if active.outline_highlight:
                    active.perm_outline_width = 3
                active.update_me = True
                return True
        self.active_ship = -1
        return False

    @property
    def active_ship_pos(self):
        if self.active_ship > -1:
            ship = self.ships[self.active_ship]
            return (ship.grid_x,ship.grid_y)
        else:
            return (-1,-1)
            
    @property
    def active_val_len(self):
        if self.active_ship > -1:
            ship = self.ships[self.active_ship]
            return len(ship.value)
        else:
            return 0
        
    def update_ships(self,circle_lock_pos,**kwargs):
        for each_ship in self.ships:
            """
            if each_ship.unit_id == self.active_ship:
                each_ship.color = each_ship.brighter
                if each_ship.outline_highlight:
                    each_ship.perm_outline_width = 3
            else:
                each_ship.color = each_ship.initcolor
                if each_ship.outline_highlight:
                    each_ship.perm_outline_width = each_ship.init_pow
            """
            each_ship.update(self,point = circle_lock_pos)

        for each_unit in self.units:
            each_unit.update(self)
        
        for each_ai in self.aiunits:
            each_ai.update(self)
            
        self.board_bg.update(self)